use std::fs::File;
use std::io::BufReader;
use crate::io::byteio::*;
use super::super::*;

const WIDTH: usize = 336;
const HEIGHT: usize = 240;

const TILE_SIZE: usize = 64;
const NUM_TILES: usize = 8192;

trait ReadClr {
    fn read_clr(&mut self) -> DecoderResult<u16>;
}

impl<T: ?Sized + ByteIO> ReadClr for T {
    fn read_clr(&mut self) -> DecoderResult<u16> {
        let clr = self.read_u16be()?;
        // BGR swap
        Ok((clr & 0x83E0) | ((clr & 0x7C00) >> 10) | ((clr & 0x1F) << 10))
    }
}

struct Plane {
    pal:        [u16; 768],
    tidc:       Vec<u16>,
    cur_w:      usize,
    cur_h:      usize,
    frame:      Vec<u16>,
}

impl Plane {
    fn new() -> Self {
        Self {
            pal:    [0; 768],
            tidc:   Vec::with_capacity(WIDTH * HEIGHT / TILE_SIZE),
            cur_w:  0,
            cur_h:  0,
            frame:  vec![0x8000; WIDTH * HEIGHT],
        }
    }
    fn unpack_data(&mut self, src: &[u8], cur_w: usize, cur_h: usize) -> DecoderResult<usize> {
        let size = unpack_sprite_data(src, &mut self.tidc)
                .map_err(|_| DecoderError::InvalidData)?;
        validate!(self.tidc.len() >= cur_w * cur_h);
        self.cur_w = cur_w;
        self.cur_h = cur_h;
        Ok(size)
    }
    fn render(&mut self, all_tiles: &[u8]) {
        if self.cur_w == 0 || self.cur_h == 0 {
            return;
        }
        for (strip, tiles) in self.frame.chunks_exact_mut(WIDTH * 8)
                .zip(self.tidc.chunks_exact(self.cur_w)).take(self.cur_h) {
            for (x, &tidx) in tiles.iter().take(WIDTH / 8).enumerate() {
                if tidx == 0 {
                    continue;
                }
                let tile = &all_tiles[(tidx & 0x1FFF) as usize * TILE_SIZE..];
                for (dline, srow) in strip.chunks_exact_mut(WIDTH).zip(tile.chunks_exact(8)) {
                    for (dst, &src) in dline[x * 8..].iter_mut().zip(srow.iter()) {
                        *dst = self.pal[usize::from(src)];
                    }
                }
            }
        }
        self.cur_w = 0;
        self.cur_h = 0;
    }
}

struct YumimiCutDecoder {
    fr:         FileReader<BufReader<File>>,
    plane:      [Plane; 4],
    data:       Vec<u8>,
    tiles:      Vec<u8>,
    tile_buf:   Vec<u8>,
}

struct Ring {
    data:   [u8; 8],
    idx:    usize,
}

impl Ring {
    fn new() -> Self { Self { data: [0; 8], idx: 1 } }
    fn add(&mut self, val: u8) {
        self.data[self.idx] = val;
        self.idx += 1;
        if self.idx > 7 {
            self.idx = 1;
        }
    }
}

fn unpack_tile_data(src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<usize> {
    let mut br = MemoryReader::new_read(src);

    let ntiles = br.read_u16be()? as usize;
    dst.clear();

    let mut ring1 = Ring::new();
    let mut ring2 = Ring::new();
    let mut def_clr = 0;
    let mut clrs = [0; 2];
    'dec_loop: loop {
        let op = br.read_byte()?;
        match op >> 6 {
            0b00 => {
                let mut len = usize::from(op & 7);
                let idx = usize::from((op >> 3) & 7);

                let clr = if idx != 0 {
                        ring1.data[idx]
                    } else {
                        let clr = br.read_byte()?;
                        ring1.add(clr);
                        clr
                    };
                if len > 0 {
                    def_clr = clr;
                }
                if len == 7 {
                    len = usize::from(br.read_byte()?);
                    if len == 0 {
                        break 'dec_loop;
                    }
                }
                for _ in 0..(len + 2) {
                    dst.push(clr);
                }
            },
            0b01 => {
                if (op & 0x20) == 0 {
                    br.read_buf(&mut clrs)?;
                }
                let mut mask = op & 0x1F;
                for _ in 0..5 {
                    dst.push(clrs[if (mask & 0x10) == 0 { 0 } else { 1 }]);
                    mask <<= 1;
                }
            },
            0b10 => {
                let mode = (op >> 2) & 0xF;
                let len = (op & 3) + 1;
                match mode {
                    0 => {
                        for _ in 0..len {
                            let clr = br.read_byte()?;
                            ring2.add(clr);
                            dst.push(clr);
                        }
                    },
                    1..=7 => {
                        let mut idx = usize::from(mode);
                        for _ in 0..len {
                            dst.push(ring2.data[idx]);
                            idx += 1;
                            if idx > 7 {
                                idx = 1;
                            }
                        }
                    },
                    8 => {
                        validate!(dst.len() >= 8);
                        for _ in 0..(len + 4) {
                            let clr = dst[dst.len() - 8];
                            dst.push(clr);
                        }
                    },
                    _ => {
                        let mut idx = usize::from(mode & 7);
                        for _ in 0..len {
                            dst.push(ring1.data[idx]);
                            idx += 1;
                            if idx > 7 {
                                idx = 1;
                            }
                        }
                    },
                }
            },
            _ => {
                let mut len = usize::from(op & 7);
                let idx = usize::from((op >> 3) & 7);

                let clr = if idx != 0 {
                        ring2.data[idx]
                    } else {
                        let clr = br.read_byte()?;
                        ring2.add(clr);
                        clr
                    };
                dst.push(clr);

                if len == 7 {
                    len = usize::from(br.read_byte()?);
                    if len == 0 {
                        break 'dec_loop;
                    }
                }
                for _ in 0..=len {
                    dst.push(def_clr);
                }
            },
        }
    }
    validate!(dst.len() >= ntiles * TILE_SIZE);
    Ok(br.tell() as usize)
}

fn unpack_sprite_data(src: &[u8], dst: &mut Vec<u16>) -> DecoderResult<usize> {
    let mut br = MemoryReader::new_read(src);

    dst.clear();

    let mut top_val = 0;
    let mut def_clr = 0;
    let mut ring = [0; 4];
    let mut ring_idx = 0;
    loop {
        let op = br.read_byte()?;
        if (op & 0x80) != 0 {
            top_val = u16::from(op) << 8;
            continue;
        }
        let mode = op >> 4;
        let len = usize::from(op & 0xF);
        match mode {
            0 => {
                for _ in 0..(len + 1) {
                    let clr = top_val + u16::from(br.read_byte()?);
                    dst.push(clr);
                }
            },
            1 if len == 0 => break,
            1 | 2 => {
                if mode == 1 {
                    def_clr = br.read_byte()?;
                }
                for _ in 0..(len + 1) {
                    let clr = top_val + u16::from(def_clr);
                    dst.push(clr);
                    def_clr = def_clr.wrapping_add(1);
                }
            },
            3 => {
                let low = br.read_byte()?;
                let clr = top_val + u16::from(low);
                for _ in 0..(len + 2) {
                    dst.push(clr);
                }
                ring[ring_idx] = low;
                ring_idx = (ring_idx + 1) & 3;
            },
            _ => {
                let clr = top_val + u16::from(ring[usize::from(mode - 4)]);
                for _ in 0..(len + 2) {
                    dst.push(clr);
                }
            },
        }
    }

    Ok(br.tell() as usize)
}

impl YumimiCutDecoder {
    fn process_commands(&mut self) -> DecoderResult<bool> {
        let mut br = MemoryReader::new_read(&self.data);
        let mut had_4001 = false;
        loop {
            let cmd = br.read_u16be()?;
            match cmd {
                0x4000 => break,
                0x4001 => {
                    let _arg = br.read_u16be()?;
                    had_4001 = true;
                },
                0x4002 => { // zero some image
                    let _arg = br.read_u16be()?;
                },
                0x4004 => { // redraw plane?
                    let _arg0 = br.read_u16be()?;
                    let _arg1 = br.read_u16be()?;
                },
                0x4005 => { // set some plane offsets?
                    let _arg0 = br.read_u16be()?;
                    let _arg1 = br.read_u16be()?;
                    let _arg2 = br.read_u16be()?;
                },
                0x4006 | 0x4007 => { // fade parameters?
                    let _arg0 = br.read_u16be()?;
                    let _arg1 = br.read_u16be()?;
                },
                0x4008 => { // set some sprite parameters
                    let _arg0 = br.read_u16be()?;
                    let _arg1 = br.read_u16be()?;
                    let _arg2 = br.read_u16be()?;
                    let _arg3 = br.read_u16be()?;
                },
                0x4009 => { // add colours offset
                    let _arg0 = br.read_u16be()?;
                    let _arg1 = br.read_u16be()?;
                    let _arg2 = br.read_u16be()?;
                    let _arg3 = br.read_u16be()?;
                },
                0x400A => {
                    let _arg0 = br.read_u16be()?;
                    let _arg1 = br.read_u16be()?;
                },
                0x400B => {
                    let mode = br.read_u16be()?;
                    match mode {
                        1 => { br.read_u16be()?; },
                        2 => {},
                        3 => { br.read_skip(6)?; },
                        _ => return Err(DecoderError::InvalidData),
                    }
                },
                0x400C => {
                    let _arg0 = br.read_u16be()?;
                },
                _ => return Err(DecoderError::NotImplemented),
            }
        }
        Ok(had_4001)
    }
    fn process_tile_copy(&mut self) -> DecoderResult<()> {
        let mut br = MemoryReader::new_read(&self.data);

        let mut src_pos = 0;
        loop {
            let cmd = br.read_u16be()?;
            let idx = (cmd & 0x1FFF) as usize * TILE_SIZE;
            match cmd {
                0x8000 => break,
                0x0000..=0x3FFF => {
                    validate!(src_pos + TILE_SIZE <= self.tile_buf.len());
                    self.tiles[idx..][..TILE_SIZE]
                            .copy_from_slice(&self.tile_buf[src_pos..][..TILE_SIZE]);
                    src_pos += TILE_SIZE;
                },
                _ => {
                    let copy_len = br.read_u16be()? as usize * TILE_SIZE;
                    validate!(src_pos + copy_len <= self.tile_buf.len());
                    self.tiles[idx..][..copy_len]
                            .copy_from_slice(&self.tile_buf[src_pos..][..copy_len]);
                    src_pos += copy_len;
                }
            }
        }

        Ok(())
    }

    fn make_frame(&mut self) -> Vec<u16> {
        for plane in self.plane.iter_mut() {
            plane.render(&self.tiles);
        }
        let mut frame = self.plane[0].frame.clone();
        for plane in self.plane[1..].iter() {
            for (dst, &src) in frame.iter_mut().zip(plane.frame.iter()) {
                if (src & 0x8000) == 0 {
                    *dst = src;
                }
            }
        }
        frame
    }
}

impl InputSource for YumimiCutDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: 10,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        let mut br = &mut self.fr;

        loop {
            let ctype = br.read_u16be()?;
            let csize = br.read_u32be()? as usize;
            match ctype {
                0 => return Err(DecoderError::EOF),
                1 => {
                    self.data.resize(csize, 0);
                    br.read_buf(&mut self.data)?;
                    let output_frame = self.process_commands().map_err(|err|
                        if err == DecoderError::ShortData { DecoderError::InvalidData } else { err })?;
                    if output_frame {
                        let frame = self.make_frame();
                        return Ok((0, Frame::VideoRGB16(frame)));
                    }
                    br = &mut self.fr;
                },
                2 => {
                    self.data.resize(csize, 0);
                    br.read_buf(&mut self.data)?;
                    unpack_tile_data(&self.data, &mut self.tile_buf)
                        .map_err(|_| DecoderError::InvalidData)?;
                },
                3 => {
                    self.data.resize(csize, 0);
                    br.read_buf(&mut self.data)?;
                    self.process_tile_copy().map_err(|_| DecoderError::InvalidData)?;
                    br = &mut self.fr;
                },
                4 => {
                    validate!(csize > 8);
                    let _xoff = br.read_u16be()? as usize;
                    let pl_idx = br.read_u16be()? as usize;
                    let tile_w = br.read_u16be()? as usize;
                    let tile_h = br.read_u16be()? as usize;
                    self.data.resize(csize - 8, 0);
                    br.read_buf(&mut self.data)?;
                    self.plane[pl_idx.min(3)].unpack_data(&self.data, tile_w, tile_h)?;
                },
                5 => {
                    validate!(csize >= 6);
                    let pal_idx = br.read_u16be()? as usize;
                    let start   = br.read_u16be()? as usize;
                    let nclrs   = br.read_u16be()? as usize;
                    validate!(start + nclrs <= 256);
                    validate!(csize >= nclrs * 2 + 6);
                    for el in self.plane[pal_idx.min(3)].pal[start..][..nclrs].iter_mut() {
                        *el = br.read_clr()?;
                    }
                    br.read_skip(csize - nclrs * 2 - 6)?;
                },
                7 => {
                    validate!(csize > 8);
                    let _xoff = br.read_u16be()? as usize;
                    let pl_idx = br.read_u16be()? as usize;
                    let tile_w = br.read_u16be()? as usize;
                    let tile_h = br.read_u16be()? as usize;
                    self.data.resize(csize - 8, 0);
                    br.read_buf(&mut self.data)?;
                    self.plane[pl_idx.min(3)].unpack_data(&self.data, tile_w, tile_h)?;
                },
                8 => {
                    validate!(csize >= 2);
                    let ntiles = br.read_u16be()? as usize;
                    validate!(csize >= ntiles * TILE_SIZE + 2);
                    self.tile_buf.resize(ntiles * TILE_SIZE, 0);
                    br.read_buf(&mut self.tile_buf)?;
                    br.read_skip(csize - 2 - ntiles * TILE_SIZE)?;
                },
                _ => br.read_skip(csize)?,
            }
        }
    }
}

pub fn open_cut(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut fr = FileReader::new_read(BufReader::new(file));

    let tag = fr.read_tag()?;
    validate!(&tag == b"ACUT");

    Ok(Box::new(YumimiCutDecoder {
        fr,
        data: Vec::new(),
        tile_buf: Vec::with_capacity(NUM_TILES * TILE_SIZE),
        plane: [Plane::new(), Plane::new(), Plane::new(), Plane::new()],
        tiles: vec![0; NUM_TILES * TILE_SIZE],
    }))
}

struct YumimiAblkDecoder {
    fr:         FileReader<BufReader<File>>,
    plane:      Plane,
    data:       Vec<u8>,
    tile_buf:   Vec<u8>,
    cur_frame:  usize,
    offsets:    Vec<u32>,
}

struct NibReader<'a> {
    br:  &'a mut dyn ByteIO,
    nib: u8,
    hi:  bool,
}

impl<'a> NibReader<'a> {
    fn new(br: &'a mut dyn ByteIO) -> Self {
        Self {
            br,
            nib: 0,
            hi:  true,
        }
    }
    fn get_nib(&mut self) -> DecoderResult<u8> {
        if self.hi {
            self.nib = self.br.read_byte()?;
            self.hi = !self.hi;
            Ok(self.nib >> 4)
        } else {
            self.hi = !self.hi;
            Ok(self.nib & 0xF)
        }
    }
}

fn unpack_ablk_tiles(br: &mut dyn ByteIO, dst: &mut Vec<u8>) -> DecoderResult<()> {
    dst.clear();
    if br.peek_byte()? != 0x80 {
        loop {
            let op = br.read_byte()?;
            let mode = op >> 6;
            let len = usize::from(op & 0x3F) + 1;
            match mode {
                0b00 => {
                    for _ in 0..len*2 {
                        dst.push(0x0);
                    }
                },
                0b01 => {
                    for _ in 0..len*2 {
                        dst.push(0xF);
                    }
                },
                0b10 => {
                    if len == 1 {
                        break;
                    }
                    let val = br.read_byte()?;
                    for _ in 0..len {
                        dst.push(val >> 4);
                        dst.push(val & 0xF);
                    }
                },
                _ => {
                    for _ in 0..len {
                        let val = br.read_byte()?;
                        dst.push(val >> 4);
                        dst.push(val & 0xF);
                    }
                }
            }
        }
    } else {
        br.read_byte()?;
        let mut dec_size = br.read_u32be()? as usize * 2;
        let mut nr = NibReader::new(br);
        while dec_size > 0 {
            let nib = nr.get_nib()?;
            let n2 = usize::from(nr.get_nib()?);
            let len = if (n2 & 8) == 0 {
                    n2 + 1
                } else {
                    (n2 & 7) * 16 + usize::from(nr.get_nib()?) + 9
                };
            validate!(dec_size >= len);
            dec_size -= len;
            for _ in 0..len {
                dst.push(nib);
            }
        }
    }
    Ok(())
}

fn unpack_ablk_bkg(br: &mut dyn ByteIO, dst: &mut Vec<u16>) -> DecoderResult<()> {
    dst.clear();
    let mut saved_val = 0;
    loop {
        let op = br.read_u16be()?;
        if op == 0xFFFF {
            break;
        }
        if dst.len() >= 5120 {
            break;
        }
        match op >> 14 {
            0b00 => {
                dst.push(op);
            },
            0b01 => {
                let len = (op & 0xFF) as usize;
                for _ in 0..=len {
                    dst.push(saved_val);
                }
            },
            0b10 => {
                let len = usize::from(br.read_byte()?);
                for _ in 0..=len {
                    dst.push(op & 0x3FFF);
                }
                saved_val = op & 0x3FFF;
            },
            _ => {
                let len = usize::from(br.read_byte()?);
                let mut val = op & 0x3FFF;
                for _ in 0..=len {
                    dst.push(val);
                    val += 1;
                }
            }
        }
    }
    Ok(())
}

fn blit_ablk_bkg(tidc: &[u16], frame: &mut [u16], tiles: &[u8], pal: &[u16; 64]) -> DecoderResult<()> {
    for (x, tcol) in tidc.chunks_exact(64).take(WIDTH / 8).enumerate() {
        for (strip, &tidx) in frame.chunks_exact_mut(WIDTH * 8).zip(tcol.iter()) {
            let tile_idx = tidx as usize;
            validate!((tile_idx + 1) * TILE_SIZE <= tiles.len());
            let tile = &tiles[tile_idx * TILE_SIZE..][..TILE_SIZE];
            for (line, row) in strip.chunks_exact_mut(WIDTH).zip(tile.chunks_exact(8)) {
                for (dst, &src) in line[x * 8..].iter_mut().zip(row.iter()) {
                    if src != 0 {
                        let clr = pal[usize::from(src)];
                        *dst = clr;
                    }
                }
            }
        }
    }
    Ok(())
}

impl InputSource for YumimiAblkDecoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: 10,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frame + 1 >= self.offsets.len() {
            return Err(DecoderError::EOF);
        }

        let br = &mut self.fr;
        let size = ((self.offsets[self.cur_frame + 1] & 0xFFFFFF) - (self.offsets[self.cur_frame] & 0xFFFFFF)) as usize;
        self.cur_frame += 1;
        validate!(size >= 16);
        validate!(size as u32 <= self.offsets[self.offsets.len() - 1]);

        self.data.resize(size, 0);
        br.read_buf(&mut self.data)?;
        validate!(&self.data[..2] == b"GI");
        let mut br = MemoryReader::new_read(&self.data);

        validate!(matches!(self.data[3], 8 | 16));

        br.read_skip(8)?;
        let width = br.read_u16be()? as usize;
        let height = br.read_u16be()? as usize;
        validate!(width > 0 && width <= WIDTH && (width & 7) == 0);
        validate!(height > 0 && height <= HEIGHT && (height & 7) == 0);
        br.read_skip(4)?;
        validate!(br.left() > 6);
        if self.data[3] == 8 {
            let start    = br.read_u16be()? as usize;
            let nclrs    = br.read_u16be()? as usize;
            validate!(start + nclrs <= 256);
            for el in self.plane.pal[start..][..nclrs].iter_mut() {
                *el = br.read_clr()?;
            }

            let start = br.tell() as usize;
            let size = self.plane.unpack_data(&self.data[start..], width / 8, height / 8)?;
            unpack_tile_data(&self.data[start + size..], &mut self.tile_buf)
                        .map_err(|_| DecoderError::InvalidData)?;
            self.plane.render(&self.tile_buf);

            Ok((0, Frame::VideoRGB16(self.plane.frame.clone())))
        } else {
            validate!(self.data.len() == width * height * 2 + 16);
            let mut frame = vec![0; WIDTH * HEIGHT];
            let cur_w = WIDTH.min(width);
            for strip in frame.chunks_exact_mut(WIDTH * 8).take(height / 8) {
                for x in (0..cur_w).step_by(8) {
                    for line in strip.chunks_exact_mut(WIDTH) {
                        for el in line[x..][..8].iter_mut() {
                            *el = br.read_clr()?;
                        }
                    }
                }
                if width > WIDTH {
                    br.read_skip((width - WIDTH) * 8)?;
                }
            }
            Ok((0, Frame::VideoRGB16(frame)))
        }
    }
}

struct Sprite {
    xpos:   usize,
    ypos:   usize,
    width:  usize,
    height: usize,
    data:   Vec<u16>,
}

impl Sprite {
    fn read_sprite(br: &mut dyn ByteIO) -> DecoderResult<Self> {
        let xpos = br.read_u16be()? as usize;
        let ypos = br.read_u16be()? as usize;
        let width = br.read_u16be()? as usize;
        let height = br.read_u16be()? as usize;
        validate!((width | height) & 7 == 0);
        let mut data = Vec::with_capacity(width * height / TILE_SIZE);
        for _ in 0..(width * height / TILE_SIZE) {
            data.push(br.read_u16be()?);
        }
        Ok(Self{ xpos, ypos, width, height, data })
    }
    fn blit(&self, frame: &mut [u16], tiles: &[u8], pal: &[u16; 64]) -> DecoderResult<()> {
        if self.xpos < WIDTH && self.ypos < HEIGHT {
            for (tx, col) in self.data.chunks_exact(self.height / 8).enumerate() {
                if self.xpos + tx * 8 >= WIDTH {
                    continue;
                }
                for (ty, &tidx) in col.iter().enumerate() {
                    if self.ypos + ty * 8 >= HEIGHT {
                        continue;
                    }
                    let tile_idx = tidx as usize;
                    validate!((tile_idx + 1) * TILE_SIZE <= tiles.len());
                    let tile = &tiles[tile_idx * TILE_SIZE..][..TILE_SIZE];
                    for (line, row) in frame.chunks_exact_mut(WIDTH).skip(self.ypos + ty * 8)
                            .zip(tile.chunks(8)) {
                        for (dst, &src) in line[self.xpos + tx * 8..].iter_mut().zip(row.iter()) {
                            if src != 0 {
                                let clr = pal[usize::from(src) + 0x10];
                                *dst = clr;
                            }
                        }
                    }
                }
            }
        }
        Ok(())
    }
}

struct YumimiAblk2Decoder {
    fr:             FileReader<BufReader<File>>,
    pal64:          [u16; 64],
    tile_buf:       Vec<u8>,
    sprites:        Vec<Sprite>,
    backgrounds:    Vec<Vec<u16>>,
    cur_frame:      usize,
}

impl InputSource for YumimiAblk2Decoder {
    fn get_num_streams(&self) -> usize { 1 }
    fn get_stream_info(&self, stream_no: usize) -> StreamInfo {
        match stream_no {
            0 => StreamInfo::Video(VideoInfo{
                    width:  WIDTH,
                    height: HEIGHT,
                    bpp:    15,
                    tb_num: 1,
                    tb_den: 10,
                 }),
            _ => StreamInfo::None
        }
    }
    fn decode_frame(&mut self) -> DecoderResult<(usize, Frame)> {
        if self.cur_frame >= self.backgrounds.len() + self.sprites.len() {
            return Err(DecoderError::EOF);
        }

        let mut frame = vec![0; WIDTH * HEIGHT];
        if self.cur_frame < self.backgrounds.len() {
            blit_ablk_bkg(&self.backgrounds[self.cur_frame], &mut frame, &self.tile_buf, &self.pal64)?;
        } else {
            blit_ablk_bkg(&self.backgrounds[0], &mut frame, &self.tile_buf, &self.pal64)?;
            self.sprites[self.cur_frame - self.backgrounds.len()].blit(&mut frame, &self.tile_buf, &self.pal64)?;
        }

        self.cur_frame += 1;
        Ok((0, Frame::VideoRGB16(frame)))
    }
}

pub fn open_ablk(name: &str) -> DecoderResult<Box<dyn InputSource>> {
    let file = File::open(name).map_err(|_| DecoderError::InputNotFound(name.to_owned()))?;
    let mut br = FileReader::new_read(BufReader::new(file));

    let tag = br.read_tag()?;
    validate!(&tag == b"ABLK");

    br.seek(SeekFrom::End(0))?;
    let tot_size = br.tell() as u32;
    br.seek(SeekFrom::Start(4))?;

    let first_offset = br.read_u32be()?;
    validate!(first_offset > 4 && (first_offset & 3) == 0 && first_offset < 0x1000);
    let noffsets = (first_offset / 4 - 1) as usize;
    let mut offsets = Vec::with_capacity(noffsets + 1);
    let mut prev_offset = first_offset;
    offsets.push(first_offset);
    for _ in 0..noffsets {
        let offset = br.read_u32be()?;
        validate!((offset & 0xFFFFFF) > (prev_offset & 0xFFFFFF));
        offsets.push(offset);
        prev_offset = offset;
    }
    validate!((offsets[offsets.len() - 1] >> 28) == 0xF);
    validate!((offsets[offsets.len() - 2] & 0xFFFFFF) < tot_size - 4);
    let idx = offsets.len() - 1;
    offsets[idx] = tot_size - 4;
    if br.peek_u16be()? == u16::from(b'G') * 256 + u16::from(b'I') {
        Ok(Box::new(YumimiAblkDecoder {
            fr: br,
            data: Vec::new(),
            tile_buf: Vec::with_capacity(NUM_TILES * TILE_SIZE),
            plane: Plane::new(),
            cur_frame: 0,
            offsets,
        }))
    } else {
        const CLR4BIT: [u16; 8] = [ 0, 4, 9, 13, 18, 22, 27, 31 ];

        let mut pal64 = [0; 64];
        for el in pal64.iter_mut() {
            let clr = br.read_u16be()? as usize;
            *el = (CLR4BIT[clr & 7] << 10) | (CLR4BIT[(clr >> 4) & 7] << 5) | CLR4BIT[(clr >> 8) & 7];
        }

        let ntiles = br.read_u16be()? as usize;
        let bkg_count = br.read_u16be()? as usize + 1;
        let sprite_count = br.read_u16be()? as usize;

        let mut tile_buf = Vec::new();
        unpack_ablk_tiles(&mut br, &mut tile_buf).map_err(|_| DecoderError::InvalidData)?;
        validate!(tile_buf.len() >= ntiles * TILE_SIZE);

        let mut backgrounds = Vec::with_capacity(bkg_count);
        for _ in 0..bkg_count {
            let mut tidc = Vec::new();
            unpack_ablk_bkg(&mut br, &mut tidc).map_err(|_| DecoderError::InvalidData)?;
            backgrounds.push(tidc);
        }

        let mut sprites = Vec::with_capacity(sprite_count);
        for _ in 0..sprite_count {
            let sprite = Sprite::read_sprite(&mut br).map_err(|_| DecoderError::InvalidData)?;
            sprites.push(sprite);
        }

        let cmd_size = br.read_u16be()? as usize;
        br.read_skip(cmd_size)?;

        Ok(Box::new(YumimiAblk2Decoder {
            fr: br,
            pal64, tile_buf, sprites,
            backgrounds,
            cur_frame: 0,
        }))
    }
}
