use crate::input::{DecoderResult, DecoderError};
use crate::io::bitreader::*;
use crate::io::codebook::*;
use crate::input::util::lzss::lz_copy;

pub struct DCL {
    lit_cb:     Codebook<u8>,
    offs_cb:    Codebook<u8>,
    ext_op_cb:  Codebook<u8>,
}

fn map_idx(idx: usize) -> u8 { idx as u8 }

impl DCL {
    pub fn new() -> Self {
        let mut cbr = TableCodebookDescReader::new(LIT_CODES, LIT_BITS, map_idx);
        let lit_cb = Codebook::new(&mut cbr, CodebookMode::LSB).unwrap();
        let mut cbr = TableCodebookDescReader::new(OFFS_CODES, OFFS_BITS, map_idx);
        let offs_cb = Codebook::new(&mut cbr, CodebookMode::LSB).unwrap();
        let mut cbr = TableCodebookDescReader::new(EXT_OP_CODES, EXT_OP_BITS, map_idx);
        let ext_op_cb = Codebook::new(&mut cbr, CodebookMode::LSB).unwrap();

        Self { lit_cb, offs_cb, ext_op_cb }
    }
    pub fn blast(&self, src: &[u8], dst: &mut Vec<u8>) -> DecoderResult<()> {
        if src.len() < 4 {
            return Err(DecoderError::ShortData);
        }
        let huff_lit = src[0] != 0;
        let offset_bits = src[1];
        if src[0] > 1 || !(4..=6).contains(&offset_bits) {
            return Err(DecoderError::InvalidData);
        }

        dst.clear();

        let mut br = BitReader::new(&src[2..], BitReaderMode::LE);
        loop {
            if br.read_bool()? {
                let ext_op = br.read_cb(&self.ext_op_cb)? as usize;
                let op = (EXT_OP_BASE[ext_op] + (br.read(EXT_OP_ADD_BITS[ext_op])? as u16)) as usize;
                if op == 0x205 {
                    break;
                }

                let copy_len = op + 2;
                let offset_base = br.read_cb(&self.offs_cb)? as usize;
                let offset = if copy_len == 2 {
                        offset_base * 4 + (br.read(2)? as usize) + 1
                    } else {
                        (offset_base << offset_bits) + (br.read(offset_bits)? as usize) + 1
                    };

                let dpos = dst.len();
                for _ in 0..copy_len {
                    dst.push(0);
                }
                lz_copy(dst, dpos, offset, copy_len);
            } else {
                let c = if huff_lit { br.read_cb(&self.lit_cb)? } else { br.read(8)? as u8 };
                dst.push(c);
            }
        }

        Ok(())
    }
}

const EXT_OP_BASE: [u16; 16] = [ 0, 1, 2, 3, 4, 5, 6, 7, 8, 10, 14, 22, 38, 70, 134, 262 ];
const EXT_OP_ADD_BITS: [u8; 16] = [ 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8 ];

const EXT_OP_CODES: &[u8; 16] = &[ 5, 3, 1, 6, 10, 2, 12, 20, 4, 24, 8, 48, 16, 32, 64, 0 ];
const EXT_OP_BITS: &[u8; 16] = &[ 3, 2, 3, 3, 4, 4, 4, 5, 5, 5, 5, 6, 6, 6, 7, 7 ];

const OFFS_CODES: &[u8; 64] = &[
      3,  13,   5, 25,   9, 17,   1, 62,  30, 46,  14, 54,  22, 38,   6, 58,
	 26,  42,  10, 50,  18, 34,  66,  2, 124, 60,  92, 28, 108, 44,  76, 12,
	116,  52,  84, 20, 100, 36,  68,  4, 120, 56,  88, 24, 104, 40,  72,  8,
	240, 112, 176, 48, 208, 80, 144, 16, 224, 96, 160, 32, 192, 64, 128,  0
];
const OFFS_BITS: &[u8; 64] = &[
    2, 4, 4, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8
];

const LIT_CODES: &[u16; 256] = &[
    1168, 4064, 2016, 3040,  992, 3552, 1504, 2528,  480,  184,   98, 3808, 1760,   34, 2784,  736,	
	3296, 1248, 2272,  224, 3936, 1888, 2912,  864, 3424, 1376, 4672, 2400,  352, 3680, 1632, 2656,
	  15,  592,   56,  608,   80, 3168,  912,  216,   66,    2,   88,  432,  124,   41,   60,  152,
	  92,    9,   28,  108,   44,   76,   24,   12,  116,  232,  104, 1120,  144,   52,  176, 1808,
	2144,   49,   84,   17,   33,   23,   20,  168,   40,    1,  784,  304,   62,  100,   30,   46,
	  36, 1296,   14,   54,   22,   68,   48,  200,  464,  208,  272,   72, 1552,  336,   96,  136,
	4000,    7,   38,    6,   58,   27,   26,   42,   10,   11,  528,    4,   19,   50,    3,   29,
	  18,  400,   13,   21,    5,   25,    8,  120,  240,  112,  656, 1040,   16, 1952, 2976,  928,
	 576, 7232, 3136, 5184, 1088, 6208, 2112, 4160,   64, 8064, 3968, 6016, 1920, 7040, 2944, 4992,
	 896, 7552, 3456, 5504, 1408, 6528, 2432, 4480,  384, 7808, 3712, 5760, 1664, 6784, 2688, 4736,
	 640, 7296, 3200, 5248, 1152, 6272, 2176, 4224,  128, 7936, 3840, 5888, 1792, 6912, 2816, 4864,
	3488, 1440, 2464,  416, 3744, 1696, 2720,  672, 3232, 1184, 2208,  160, 3872, 1824, 2848,  800,
	3360, 1312, 2336,  288, 3616, 1568, 2592,  544, 3104, 1056, 2080,   32, 4032, 1984, 3008,  960,
	3520, 1472, 2496,  448, 3776, 1728, 2752,  704, 3264, 1216, 2240,  192, 3904, 1856, 2880,  832,
	 768, 3392, 7424, 3328, 5376, 1344, 1280, 6400, 2304, 2368, 4352,  256, 7680, 3584,  320, 5632,
	1536, 6656, 3648, 1600, 2624, 2560, 4608,  512, 7168, 3072, 5120, 1024, 6144, 2048, 4096,    0
];

const LIT_BITS: &[u8; 256] = &[
	11, 12, 12, 12, 12, 12, 12, 12, 12,  8,  7, 12, 12,  7, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 13, 12, 12, 12, 12, 12,
	 4, 10,  8, 12, 10, 12, 10,  8,  7,  7,  8,  9,  7,  6,  7,  8,
	 7,  6,  7,  7,  7,  7,  8,  7,  7,  8,  8, 12, 11,  7,  9, 11,
	12,  6,  7,  6,  6,  5,  7,  8,  8,  6, 11,  9,  6,  7,  6,  6,
	 7, 11,  6,  6,  6,  7,  9,  8,  9,  9, 11,  8, 11,  9, 12,  8,
	12,  5,  6,  6,  6,  5,  6,  6,  6,  5, 11,  7,  5,  6,  5,  5,
	 6, 10,  5,  5,  5,  5,  8,  7,  8,  8, 10, 11, 11, 12, 12, 12,
	13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
	13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
	12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
	12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
	13, 12, 13, 13, 13, 12, 13, 13, 13, 12, 13, 13, 13, 13, 12, 13,
	13, 13, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13
];
