use crate::input::*;
use crate::io::byteio::*;

/// Copies requested amount of bytes from previous position in the same buffer.
/// If source area overlaps with destination area already copied values should be used e.g. copying with offset 1 means essentially to repeat previous byte requested number of times.
pub fn lz_copy(buf: &mut [u8], dst_pos: usize, offset: usize, len: usize) {
    if dst_pos < offset {
        panic!("Copy offset is before buffer start.");
    }
    let ipos = dst_pos - offset;
    let buf = &mut buf[ipos..];
    if ipos + len <= dst_pos {
        let (src, dst) = buf.split_at_mut(offset);
        dst[..len].copy_from_slice(&src[..len]);
    } else {
        for i in 0..len {
            buf[offset + i] = buf[i];
        }
    }
}

/// Decompress a flavour of LZSS
pub fn lzss_unpack(src: &[u8], dst: &mut [u8]) -> DecoderResult<()> {
    lzss_unpack_ext(src, dst, 0x20)
}

/// Decompress a flavour of LZSS with dictionary pre-filled using custom value
pub fn lzss_unpack_ext(src: &[u8], dst: &mut [u8], dict_val: u8) -> DecoderResult<()> {
    let mut br = MemoryReader::new_read(src);
    let mut bw = MemoryWriter::new_write(dst);

    let mut window = [dict_val; 4096];
    let mut wpos = 4096 - 18;
    let mut flags = 0;
    let mut fbits = 0;
    while br.left() > 0 {
        if fbits == 0 {
            flags = br.read_byte()?;
            fbits = 8;
        }
        if (flags & 1) != 0 {
            let b = br.read_byte()?;
            window[wpos] = b;
            wpos = (wpos + 1) & 0xFFF;
            bw.write_byte(b)?;
        } else {
            let lo_byte = br.read_byte()?;
            let hi_byte = br.read_byte()?;

            let offset = (u16::from(hi_byte & 0xF0) * 16 + u16::from(lo_byte)) as usize;
            let len = ((hi_byte & 0xF) + 3) as usize;

            for i in 0..len {
                let b = window[(offset + i) & 0xFFF];
                window[(wpos + i) & 0xFFF] = b;
                bw.write_byte(b)?;
            }
            wpos = (wpos + len) & 0xFFF;
        }
        flags >>= 1;
        fbits  -= 1;
    }
    let used = bw.tell();
    let dsize = dst.len() as u64;
    if used == dsize {
        Ok(())
    } else {
println!("LZSS: in used {}/{} out {}/{}", br.tell(), src.len(), used, dsize);
        Err(DecoderError::InvalidData)
    }
}
