use std::fs;
use std::fs::File;
use std::io;
use std::path::Path;
use std::ffi::OsString;

fn file_or_dir_exists(path: &Path) -> bool {
    File::open(path).is_ok()
}

pub fn find_path<P: AsRef<Path>>(path: P) -> std::io::Result<OsString> {
    find_path_int(path.as_ref())
}

fn find_path_int(path: &Path) -> std::io::Result<OsString> {
    if file_or_dir_exists(path) {
        Ok(OsString::from(path))
    } else {
        let (dname, fname) = if let Some(dirname) = path.parent() {
                let dir_name = if dirname.as_os_str().is_empty() { Path::new(".") } else { dirname };
                (dir_name, path.strip_prefix(dirname).unwrap())
            } else {
                (Path::new("."), path)
            };
        let fname = if fname.as_os_str().is_empty() {
                path.to_str().unwrap()
            } else if let Some(nm) = fname.to_str() {
                nm
            } else {
                return Err(io::Error::from(io::ErrorKind::InvalidData));
            };

        if let Ok(dirname) = find_path_int(dname) {
            let fname = fname.to_ascii_lowercase();
            let prefix = dirname.clone();

            let dir = fs::read_dir(dirname).unwrap();
            for dirent in dir.flatten() {
                if let Some(cur_name) = dirent.file_name().to_str() {
                    if cur_name.to_ascii_lowercase() == fname {
                        let mut full_name = std::ffi::OsString::new();
                        full_name.push(prefix);
                        full_name.push(std::path::MAIN_SEPARATOR_STR);
                        full_name.push(cur_name);
                        return Ok(full_name);
                    }
                }
            }
        }
        Err(io::Error::from(io::ErrorKind::NotFound))
    }
}

pub fn open_file_igncase<P: AsRef<Path>>(name: P) -> std::io::Result<File> {
    if let Ok(path) = find_path(name) {
        if let Ok(file) = File::open(path) {
            return Ok(file)
        }
    }
    Err(io::Error::from(io::ErrorKind::NotFound))
}
