pub struct YUV2RGB {
    u_tab1: [i16; 256],
    u_tab2: [i16; 256],
    v_tab1: [i16; 256],
    v_tab2: [i16; 256],
    coeffs: [f32; 9],
}

pub const DEFAULT_COEFFS: [f32; 9] = [
    1.0,  0.0,      1.13983,
    1.0, -0.39465, -0.5806,
    1.0,  2.03211,  0.0
];

impl YUV2RGB {
    pub fn new() -> Self { Self::new_with_coeffs(&DEFAULT_COEFFS) }
    pub fn new_with_coeffs(coeffs: &[f32; 9]) -> Self {
        let u_tab1 = std::array::from_fn(|i| ((i as f32 - 128.0) * coeffs[4]) as i16);
        let u_tab2 = std::array::from_fn(|i| ((i as f32 - 128.0) * coeffs[7]) as i16);
        let v_tab1 = std::array::from_fn(|i| ((i as f32 - 128.0) * coeffs[2]) as i16);
        let v_tab2 = std::array::from_fn(|i| ((i as f32 - 128.0) * coeffs[5]) as i16);

        Self { coeffs: *coeffs, u_tab1, u_tab2, v_tab1, v_tab2 }
    }
    pub fn convert(&self, y: u8, u: u8, v: u8, dst: &mut [u8]) {
        let luma = i16::from(y);
        dst[0] = (luma + self.v_tab1[usize::from(v)]).clamp(0, 255) as u8;
        dst[1] = (luma + self.v_tab2[usize::from(v)] + self.u_tab1[usize::from(u)]).clamp(0, 255) as u8;
        dst[2] = (luma + self.u_tab2[usize::from(u)]).clamp(0, 255) as u8;
    }
    pub fn generate_lut15(coeffs: &[f32; 9], swapuv: bool) -> [u16; 32768] {
        std::array::from_fn(|i| {
            let y = (i >> 10) as f32;
            let (u, v) = if !swapuv {
                    (((i >> 5) & 0x1F) as f32 - 16.0, (i & 0x1F) as f32 - 16.0)
                } else {
                    ((i & 0x1F) as f32 - 16.0, ((i >> 5) & 0x1F) as f32 - 16.0)
                };

            let r = y * coeffs[0] + u * coeffs[1] + v * coeffs[2];
            let g = y * coeffs[3] + u * coeffs[4] + v * coeffs[5];
            let b = y * coeffs[6] + u * coeffs[7] + v * coeffs[8];

            ((r.max(0.0).min(31.0) as u16) << 10) |
              ((g.max(0.0).min(31.0) as u16) << 5) |
               (b.max(0.0).min(31.0) as u16)
        })
    }
}
