#[allow(dead_code)]
#[allow(clippy::upper_case_acronyms)]
mod io;
#[allow(dead_code)]
#[allow(clippy::upper_case_acronyms)]
mod input;
use input::*;
mod output;
use output::*;

#[allow(unused_assignments)]
fn main() {
    let args: Vec<_> = std::env::args().collect();

    if args.len() < 2 || &args[1] == "-h" || &args[1] == "--help" {
        println!("Usage:");
        println!("for decoding game video and audio files:");
        println!("  na_game_tool [-ifmt input_format] input [-ofmt output_format] output");
        println!("  where 'input' is an input name opened with 'input_format' plugin");
        println!("  if 'input_format' is not provided the format will be guessed (poorly)");
        println!("  use na_game_tool --list-input-formats to get a full list of input plugins");
        println!();
        println!("  'output' is a name of output AVI file, image sequence or WAV file");
        println!("  supported types are:");
        println!("  -ofmt avi output.avi (writing raw image AVI)");
        println!("  -ofmt imgseq output%d.ppm (writing a sequence of images using template name)");
        println!("  -ofmt wav output.wav (writing output audio)");
        println!("  if '-ofmt' is not provided, output format should be guessed");
        println!("  use na_game_tool --list-output-formats to get a full list of output writers");
        println!();
        println!();
        println!("for extracting game archives:");
        println!("  na_game_tool [-extract] [-ifmt input_format] input outputdir");
        println!("  where 'input' is an input name opened with 'input_format' plugin");
        println!("  if 'input_format' is not provided the format will be guessed (poorly)");
        println!("  if '-extract' switch is not provided the tool may try to decode input instead");
        println!("  use na_game_tool --list-archive-formats to get a full list of input plugins");
        println!();
        println!("  'outputdir' is a name of output directory (use 'null' for test)");
        return;
    }

    if &args[1] == "-v" || &args[1] == "--version" {
        println!("na_game_tool version {}", env!("CARGO_PKG_VERSION"));
        return;
    }

    if &args[1] == "--list-input-formats" {
        println!("Supported input plugins:");
        for entry in MMEDIA_PLUGINS.iter() {
            println!("  {:10} {}", entry.name, entry.long_name);
        }
        return;
    }

    if &args[1] == "--list-output-formats" {
        println!("Supported output writers:");
        for entry in OUTPUT_PLUGINS.iter() {
            println!("  {:10} {}", entry.name, entry.long_name);
        }
        return;
    }

    if &args[1] == "--list-archive-formats" {
        println!("Supported archive extractors:");
        for entry in ARCHIVE_PLUGINS.iter() {
            println!("  {:10} {}", entry.name, entry.long_name);
        }
        return;
    }

    let mut arg_pos = 1;
    let mut ifmt_idx = 0;
    let mut iname_idx = 0;
    let mut ofmt_idx = 0;
    let mut oname_idx = 0;
    let mut no_audio = false;
    let mut do_extract = false;
    let mut no_convert = false;

    if &args[arg_pos] == "-extract" {
        do_extract = true;
        arg_pos += 1;
    }

    if &args[arg_pos] == "-ifmt" || &args[arg_pos] == "-f" {
        if arg_pos + 1 >= args.len() {
            println!("no input format name provided");
            std::process::exit(1);
        }
        ifmt_idx = arg_pos + 1;
        arg_pos += 2;
    }

    if arg_pos >= args.len() {
        println!("input name is not provided!");
        return;
    }
    iname_idx = arg_pos;
    arg_pos += 1;

    if arg_pos < args.len() && (&args[arg_pos] == "-ofmt" || &args[arg_pos] == "-f" ) {
        if arg_pos + 1 >= args.len() {
            println!("no output format name provided");
            std::process::exit(1);
        }
        ofmt_idx = arg_pos + 1;
        arg_pos += 2;
    }

    if arg_pos >= args.len() {
        println!("output name is not provided!");
        return;
    }
    oname_idx = arg_pos;
    arg_pos += 1;

    if arg_pos < args.len() && &args[arg_pos] == "-an" {
        no_audio = true;
        arg_pos += 1;
    }

    if arg_pos < args.len() && &args[arg_pos] == "-nocvt" {
        no_convert = true;
        arg_pos += 1;
    }

    println!("trying to encode {} format {} to {} format {}", args[iname_idx], if ifmt_idx > 0 { &args[ifmt_idx] } else { "???" }, args[oname_idx], if ofmt_idx > 0 { &args[ofmt_idx] } else if do_extract { "output_dir" } else { "???"});

    let ret = if ifmt_idx > 0 {
            open_input(&args[ifmt_idx], &args[iname_idx], do_extract)
        } else {
            open_unknown_input(&args[iname_idx], do_extract)
        };
    let inputt = match ret {
            Ok(val) => val,
            Err(err) => {
                println!("error opening input: {err}");
                std::process::exit(1);
            },
        };
    if do_extract && !matches!(inputt, InputType::Archive(_)) {
        println!("input format is not archive");
        std::process::exit(1);
    }

    match inputt {
        InputType::Multimedia(mut input) => {
            let ret = if ofmt_idx > 0 {
                    open_output(&args[ofmt_idx], &args[oname_idx])
                } else {
                    open_output_auto(&args[oname_idx])
                };
            let mut output = match ret {
                    Ok(val) => val,
                    Err(err) => {
                        println!("error opening output: {err}");
                        std::process::exit(1);
                    },
                };

            let nstreams = input.get_num_streams();
            let mut ostreams = 0;
            let mut stream_map = Vec::with_capacity(nstreams);
            for stream_no in 0..nstreams {
                let info = input.get_stream_info(stream_no);
                println!("  stream {}: {:?}", stream_no, info);
                if let (true, StreamInfo::Audio(_)) = (no_audio, info) {
                    println!("    (skipped)");
                    stream_map.push(None);
                    continue;
                }
                match output.add_stream(ostreams, info) {
                    Ok(()) => {
                        stream_map.push(Some(ostreams));
                        ostreams += 1;
                    },
                    Err(EncoderError::Ignored) => {
                        stream_map.push(None);
                    },
                    Err(err) => {
                        println!("error adding stream: {err}");
                        std::process::exit(2);
                    }
                }
            }

            if let Err(err) = output.finish_header() {
                println!("error writing header: {err}");
                std::process::exit(3);
            }

            loop {
                match input.decode_frame() {
                    Ok((stream_no, frame)) => {
                        if stream_no >= stream_map.len() {
                            println!("got a frame for unregistered stream");
                            std::process::exit(5);
                        }
                        if let Some(ostream_no) = stream_map[stream_no] {
                            if let Err(err) = output.write(ostream_no, frame) {
                                println!("error writing frame: {err}");
                                std::process::exit(4);
                            }
                        }
                    },
                    Err(DecoderError::EOF) => break,
                    Err(err) => {
                        println!("error getting new frame: {err}");
                        break;
                    },
                }
            }

            if let Err(err) = output.finish() {
                println!("error finalising output: {err}");
                std::process::exit(5);
            }
        },
        InputType::Archive(mut input) => {
            if no_convert {
                input.set_no_convert();
            }
            let ret = output::dir::DirectoryWriter::create(&args[oname_idx]);
            let mut dirw = match ret {
                    Ok(val) => val,
                    Err(err) => {
                        println!("error opening output directory: {err}");
                        std::process::exit(1);
                    },
                };

            loop {
                match input.get_file_name() {
                    Ok(name) => {
                        println!("  extracting {name}");
                        let ret = dirw.get_writer(&name);
                        let mut wr = match ret {
                                Ok(val) => val,
                                Err(err) => {
                                    println!("error opening output file: {err}");
                                    break;
                                },
                            };
                        match input.extract_file(wr.as_mut()) {
                            Ok(()) => {},
                            Err(err) => {
                                println!("error extracting output file: {err}");
                                break;
                            },
                        }
                        if wr.flush().is_err() {
                            println!("error finalising output");
                            std::process::exit(5);
                        }
                    },
                    Err(DecoderError::EOF) => break,
                    Err(err) => {
                        println!("error reading archive: {err}");
                        break;
                    },
                }
            }
        },
    }
}
