use std::fs::File;
use std::io::BufWriter;
use std::path;
use crate::io::byteio::*;

struct NullWriter {}

impl ByteIO for NullWriter {
    fn read_buf(&mut self, _buf: &mut [u8]) -> ByteIOResult<()> { Err(ByteIOError::NotImplemented) }
    fn read_buf_some(&mut self, _buf: &mut [u8]) -> ByteIOResult<usize> { Err(ByteIOError::NotImplemented) }
    fn peek_buf(&mut self, _buf: &mut [u8]) -> ByteIOResult<usize> { Err(ByteIOError::NotImplemented) }
    fn read_byte(&mut self) -> ByteIOResult<u8> { Err(ByteIOError::NotImplemented) }
    fn peek_byte(&mut self) -> ByteIOResult<u8> { Err(ByteIOError::NotImplemented) }
    fn write_buf(&mut self, _buf: &[u8]) -> ByteIOResult<()> { Ok(()) }
    fn tell(&mut self) -> u64 { 0 }
    fn seek(&mut self, _pos: SeekFrom) -> ByteIOResult<u64> { Ok(0) }
    fn is_eof(&self) -> bool { true }
    fn is_seekable(&mut self) -> bool { true }
    fn size(&mut self) -> i64 { -1 }
    fn flush(&mut self) -> ByteIOResult<()> { Ok(()) }
}

#[allow(dead_code)]
#[derive(Clone,Copy,Debug,PartialEq)]
pub enum DWError {
    OpenError,
    NotADirectory,
    InvalidName,
    NotImplemented,
}

impl std::fmt::Display for DWError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> Result<(), std::fmt::Error> {
        match self {
            DWError::OpenError => write!(f, "error opening/creating file or directory"),
            DWError::InvalidName => write!(f, "invalid input name"),
            DWError::NotADirectory => write!(f, "not a directory"),
            DWError::NotImplemented => write!(f, "unsupported feature"),
        }
    }
}

pub type DWResult<T> = Result<T, DWError>;

pub struct DirectoryWriter {
    basename:   String,
    is_null:    bool,
}

impl DirectoryWriter {
    fn mkdir(path: &str) -> DWResult<()> {
        if let Ok(file) = File::open(path) {
            if let Ok(meta) = file.metadata() {
                if meta.is_dir() {
                    return Ok(());
                }
            }
            return Err(DWError::NotADirectory);
        }
        match std::fs::create_dir(path) {
            Ok(_) => Ok(()),
            Err(_err) => Err(DWError::OpenError),
        }
    }
    fn create_dir_recursive(prefix: &str, path: &[&str]) -> DWResult<()> {
        let mut dirname = prefix.to_owned();
        for subname in path.iter() {
            if subname.is_empty() || subname == &".." {
                return Err(DWError::InvalidName);
            }
            dirname.push(std::path::MAIN_SEPARATOR);
            dirname += subname;
            Self::mkdir(&dirname)?;
        }
        Ok(())
    }
    pub fn create(path: &str) -> DWResult<Self> {
        match path {
            "" => Err(DWError::InvalidName),
            "null" => Ok(Self {
                basename:   path.to_owned(),
                is_null:    true,
            }),
            _ => {
                Self::mkdir(path)?;
                Ok(Self {
                    basename:   path.to_owned(),
                    is_null:    false,
                })
            },
        }
    }
    pub fn get_writer(&mut self, name: &str) -> DWResult<Box<dyn ByteIO>> {
        if name.is_empty() {
            return Err(DWError::InvalidName);
        }
        if !self.is_null {
            let mut filename = self.basename.clone();
            if !filename.ends_with(path::MAIN_SEPARATOR) {
                filename.push(path::MAIN_SEPARATOR);
            }

            if name.contains(path::MAIN_SEPARATOR) {
                let mut path: Vec<&str> = name.split(path::MAIN_SEPARATOR).collect();
                if let Some(last) = path.pop() {
                    if last.is_empty() {
                        return Err(DWError::InvalidName);
                    }
                }
                Self::create_dir_recursive(&filename, &path)?;
            }

            filename += name;
            match File::create(&filename) {
                Ok(file) => {
                    Ok(Box::new(FileWriter::new_write(BufWriter::new(file))))
                },
                Err(_err) => Err(DWError::OpenError),
            }
        } else {
            Ok(Box::new(NullWriter{}))
        }
    }
}
